import argparse
import re
import sys

QUAKE1 = 0
QUAKE2 = 1
QUAKE3 = 2

# global location to store/read parameters from. this is automatically populated when you call qrucible_init().

qparams = {}

def qcrucible_init():

	parser = argparse.ArgumentParser()
	parser.add_argument("--set", metavar='KEY=VALUE', action="append", type=_qrucible_kv, default=[])

	qparams.clear()
	qparams.update(dict(parser.parse_args().set))

# Finds the next available target number by looking up the highest existing target number.
#
# Returns:
#     tuple[bool, int]: A tuple containing:
#         - bool: True if successful, False if failed to find highest target
#         - int: Next available target number if successful, -1 if failed

def qrucible_get_next_target_name() -> tuple[bool, str]:
	result, highest_target = _qrucible_find_highest_target_num()

	if result:
		return True, qrucible_make_target_name(highest_target + 1)
	else:
		return False, ""

def qrucible_get_next_target_num() -> int:
	_, highest_target = _qrucible_find_highest_target_num()
	return highest_target + 1


def qrucible_make_target_name(_num) -> str:
	return qparams["target_prefix"] + str(_num)

# emits a line that tells Qrucible the plugin has finished successfully.

def qrucible_emit_success():
	print("QRUCIBLE:Success, Success")
	sys.exit(0)     # in sys stuff, 0 is success

# emits a line that tells Qrucible the plugin failed.

def qrucible_emit_failure(message):
	print("QRUCIBLE:Failure," + message)
	sys.exit(1)     # in sys stuff, non-zero is failure

# -=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=
# INTERNAL
#
# It's best practice not to call these functions as they may change in the
# future. They start with an underscore to indicate that they are not part of
# the public API.
# -=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=

def _qrucible_kv(s: str) -> tuple[str, str]:
	if "=" not in s:
		raise argparse.ArgumentTypeError("use key=value")
	k, v = s.split("=", 1)
	return k, v

def _qrucible_find_highest_target_num() -> tuple[bool, int]:
	try:
		highest_target = 0
		kv_re = re.compile(r'"\s*([^"]+)\s*"\s*"\s*([^"]*)\s*"')
		prefix = qparams["target_prefix"]
		val_re = re.compile(rf'^{re.escape(prefix)}(\d+)$')

		with open(qparams["map_name"], 'r', encoding='utf-8', errors='ignore') as f:
			for line in f:
				m = kv_re.search(line)
				if not m:
					continue

				key, value = m.group(1), m.group(2)
				if 'target' in key.lower():
					m2 = val_re.match(value)
					if not m2:
						continue
					target_num = int(m2.group(1))
					highest_target = max(highest_target, target_num)

		return True, highest_target
	except Exception:
		return False, -1


# script tips:
#
# to check for a specific game_id:
#
# 	if int(qparams["game_id"]) != QUAKE1:
# 		qrucible_emit_failure("This plugin only works for Quake 1.")
# 		return

